import asyncio
import os

import numpy as np
import matplotlib.pyplot as plt

from numpy.polynomial.polynomial import polyfit

from py_pli.pylib import VUnits
from py_pli.pylib import GlobalVar
from py_pli.pylib import send_gc_event

from config_enum import hal_enum as hal_config

from urpc_enum.measurementparameter import MeasurementParameter

from virtualunits.HAL import HAL
from virtualunits.meas_seq_generator import meas_seq_generator
from virtualunits.meas_seq_generator import TriggerSignal
from virtualunits.meas_seq_generator import OutputSignal
from virtualunits.meas_seq_generator import MeasurementChannel
from virtualunits.meas_seq_generator import IntegratorMode
from virtualunits.meas_seq_generator import AnalogControlMode

from fleming.common.firmware_util import *

hal_unit: HAL = VUnits.instance.hal
meas_unit = hal_unit.measurementUnit

fmb_endpoint = get_node_endpoint('fmb')
eef_endpoint = get_node_endpoint('eef')
meas_endpoint = get_measurement_endpoint()

report_path = hal_unit.get_config(hal_config.Application.GCReportPath)
os.makedirs(report_path, exist_ok=True)

# HV Conversion: U_x = U_adc * (200M/430k + 200M/1M3 + 1) - 3.3V * 200M/430k
# -12V Conversion: U_x = U_adc * 12.232625 - 26.757783 (ltspice) <- doesn't seem to work this way

ptb_hv_r1 = 430.0e3
ptb_hv_r2 = 1.3e6
ptb_hv_r3 = 200.0e6

ptb_hv_scaling = 2.5 * (ptb_hv_r3 / ptb_hv_r1 + ptb_hv_r3 / ptb_hv_r2 + 1)
ptb_hv_offset  = -3.3 * ptb_hv_r3 / ptb_hv_r1

ptb_control_sw_off = {
    'tp210' : {'channel':EEFAnalogInput.PTBCONTROLVIN0, 'scaling':12.25, 'offset':0},
    'tp223' : {'channel':EEFAnalogInput.PTBCONTROLVIN1, 'scaling':30.61, 'offset':-28.24},  # Determined empirical
    'tp207' : {'channel':EEFAnalogInput.PTBCONTROLVIN2, 'scaling':12.25, 'offset':0},
    'tp216' : {'channel':EEFAnalogInput.PTBCONTROLVIN3, 'scaling':12.25, 'offset':0},
}

ptb_control_sw_on = {
    'tp213' : {'channel':EEFAnalogInput.PTBCONTROLVIN0, 'scaling':12.25, 'offset':0},
    'tp221' : {'channel':EEFAnalogInput.PTBCONTROLVIN2, 'scaling':12.25, 'offset':0},
    'tp200' : {'channel':EEFAnalogInput.PTBCONTROLVIN3, 'scaling':12.25, 'offset':0},
}

ptb_cathode_sw_off = {
    'dyn5' : {'channel':EEFAnalogInput.PTBCATHODEVIN0, 'scaling':ptb_hv_scaling, 'offset':ptb_hv_offset},
    'dyn3' : {'channel':EEFAnalogInput.PTBCATHODEVIN1, 'scaling':ptb_hv_scaling, 'offset':ptb_hv_offset},
    'dyn7' : {'channel':EEFAnalogInput.PTBCATHODEVIN2, 'scaling':ptb_hv_scaling, 'offset':ptb_hv_offset},
    'dyn1' : {'channel':EEFAnalogInput.PTBCATHODEVIN3, 'scaling':ptb_hv_scaling, 'offset':ptb_hv_offset},
}

ptb_cathode_sw_on = {
    'dyn4'    : {'channel':EEFAnalogInput.PTBCATHODEVIN0, 'scaling':ptb_hv_scaling, 'offset':ptb_hv_offset},
    'dyn2'    : {'channel':EEFAnalogInput.PTBCATHODEVIN1, 'scaling':ptb_hv_scaling, 'offset':ptb_hv_offset},
    'dyn6'    : {'channel':EEFAnalogInput.PTBCATHODEVIN2, 'scaling':ptb_hv_scaling, 'offset':ptb_hv_offset},
    'cathode' : {'channel':EEFAnalogInput.PTBCATHODEVIN3, 'scaling':ptb_hv_scaling, 'offset':ptb_hv_offset},
}

ptb_anode_sw_on = {
    'anode' : {'channel':EEFAnalogInput.PTBANODEVIN0, 'scaling':2.6325, 'offset':0},                        # 2.5V * 10k53 / 10k
    'dyn8'  : {'channel':EEFAnalogInput.PTBANODEVIN1, 'scaling':ptb_hv_scaling, 'offset':ptb_hv_offset},
    '2v5'   : {'channel':EEFAnalogInput.PTBANODEVIN2, 'scaling':2.5, 'offset':0},
    'dyn9'  : {'channel':EEFAnalogInput.PTBANODEVIN3, 'scaling':ptb_hv_scaling, 'offset':ptb_hv_offset},
}

phd_hv_monitor = {'scaling':-2.589231e3, 'offset':0}     # 3.3V / 5V * 23k8 / 9k1 * -1500V
phd_hv_setting = {'scaling':-5.050505e-4, 'offset':0}    # 5V / 6.6V / -1500V

phd_voltage_average = 8

phd_voltage_reference = {
    'tp200' :  12.0,    # DC Link Voltage
    'tp207' :   3.3,    # ADC / DAC / Digital IO
    'tp210' :   5.0,    # Analog-Counting
    'tp213' :   5.0,    # V_reference
    'tp216' :   5.0,    # Counting
    'tp221' :   6.0,    # Counting
    'tp223' : -12.0,    # All Switches
    'anode' :   0.1,    # Anode Offset Voltage
}

phd_voltage_tolerance = {
    'tp200'      : 0.05,    #  5 %
    'tp207'      : 0.05,    #  5 %
    'tp210'      : 0.05,    #  5 %
    'tp213'      : 0.05,    #  5 %
    'tp216'      : 0.05,    #  5 %
    'tp221'      : 0.05,    #  5 %
    'tp223'      : 0.05,    #  5 %
    'hv_monitor' : 0.10,    # 10 %
    'cathode'    : 0.10,    # 10 %
    'dyn1'       : 0.10,    # 10 %
    'dyn2'       : 0.10,    # 10 %
    'dyn3'       : 0.10,    # 10 %
    'dyn4'       : 0.10,    # 10 %
    'dyn5'       : 0.10,    # 10 %
    'dyn6'       : 0.10,    # 10 %
    'dyn7'       : 0.10,    # 10 %
    'dyn8'       : 0.10,    # 10 %
    'dyn9'       : 0.20,    # 20 %, due to the measurement not being very reliable
    'anode'      : 0.20,    # 20 %, due to the measurement not being very reliable
}

# PMT static gating voltage divider coefficients for each dynode.
# dynode voltage = (HV + 10V) * coefficient - 10V   (due to 10V zener diode and HV being negative)
phd_dynode_coefficient = {
    'cathode' : 4440 / 5001,
    'dyn1'    : 3820 / 5001,
    'dyn2'    : 3460 / 5001,
    'dyn3'    : 3100 / 5001,
    'dyn4'    : 2740 / 5001,
    'dyn5'    : 2380 / 5001,
    'dyn6'    : 2020 / 5001,
    'dyn7'    : 1660 / 5001,
    'dyn8'    : 1300 / 5001,
    'dyn9'    :  300 / 5001,
}

phd_block_dynode_coefficient = {
    'dyn1' : 1,
    'dyn2' : 1,
    'dyn5' : (phd_dynode_coefficient['dyn3'] - phd_dynode_coefficient['dyn6']) * (2000 / 2270) + phd_dynode_coefficient['dyn6'],
}

phd_high_voltage_off_max_abs = 20   # max abs 20V for high voltage off

async def phd_test():

    GlobalVar.set_stop_gc(False)
    await send_gc_msg(f"Starting PHD Test")

    await start_firmware('fmb')
    await fmb_endpoint.SetDigitalOutput(FMBDigitalOutput.PSUON, 0)   # Base Tester PSUON is active low
    try:
        await asyncio.sleep(0.1)
        await start_firmware('eef')
        if GlobalVar.get_stop_gc():
            return f"phd_test stopped by user"

        phd_test_error = False

        # Control Voltage Tests ########################################################################################

        voltages = await phd_get_control_voltages()
        if GlobalVar.get_stop_gc():
            return f"phd_test stopped by user"

        PyLogger.logger.info(f"PHD Control Voltages:")
        PyLogger.logger.info(f"voltage    ; value     ; reference ; error")
        for voltage, value in voltages.items():
            reference = phd_voltage_reference[voltage]
            tolerance = phd_voltage_tolerance[voltage]
            error = abs((value - reference) / reference)
            PyLogger.logger.info(f"{voltage:10} ; {value:9.2f} ; {reference:9.2f} ; {error:7.2%}")
            if error > tolerance:
                phd_test_error = True
                await send_gc_msg(f"FAILED - {voltage.upper()} Voltage = {reference} V")
            else:
                await send_gc_msg(f"PASSED - {voltage.upper()} Voltage = {reference} V")

        # High Voltage Tests ###########################################################################################

        # HV = 0 V
        result = await phd_high_voltage_test(hv_target=0)
        if GlobalVar.get_stop_gc():
            return f"phd_test stopped by user"

        if result != 'PASSED':
            phd_test_error = True
            await send_gc_msg(f"FAILED - PMT High Voltage = 0 V")
        else:
            await send_gc_msg(f"PASSED - PMT High Voltage = 0 V")

        # HV = -750 V
        result = await phd_high_voltage_test(hv_target=-750)
        if GlobalVar.get_stop_gc():
            return f"phd_test stopped by user"

        if result != 'PASSED':
            phd_test_error = True
            await send_gc_msg(f"FAILED - PMT High Voltage = -750 V")
        else:
            await send_gc_msg(f"PASSED - PMT High Voltage = -750 V")

        # HV = -1500 V
        result = await phd_high_voltage_test(hv_target=-1500)
        if GlobalVar.get_stop_gc():
            return f"phd_test stopped by user"

        if result != 'PASSED':
            phd_test_error = True
            await send_gc_msg(f"FAILED - PMT High Voltage = -1500 V")
        else:
            await send_gc_msg(f"PASSED - PMT High Voltage = -1500 V")

        # PMT Counting Measurement Tests ###############################################################################

        result = await phd_counting_measurment_test()
        if GlobalVar.get_stop_gc():
            return f"phd_test stopped by user"

        if result != 'PASSED':
            phd_test_error = True
            await send_gc_msg(f"FAILED - PMT Counting Measurement Test")
        else:
            await send_gc_msg(f"PASSED - PMT Counting Measurement Test")

        # PMT Analog Measurement Tests #################################################################################

        result = await phd_analog_measurement_test(mode='auto')
        if GlobalVar.get_stop_gc():
            return f"phd_test stopped by user"

        if result != 'PASSED':
            phd_test_error = True
            await send_gc_msg(f"FAILED - PMT Analog Auto Range Test")
        else:
            await send_gc_msg(f"PASSED - PMT Analog Auto Range Test")

        result = await phd_analog_measurement_test(mode='low')
        if GlobalVar.get_stop_gc():
            return f"phd_test stopped by user"

        if result != 'PASSED':
            phd_test_error = True
            await send_gc_msg(f"FAILED - PMT Analog Low Range Test")
        else:
            await send_gc_msg(f"PASSED - PMT Analog Low Range Test")

        result = await phd_analog_measurement_test(mode='high')
        if GlobalVar.get_stop_gc():
            return f"phd_test stopped by user"

        if result != 'PASSED':
            phd_test_error = True
            await send_gc_msg(f"FAILED - PMT Analog High Range Test")
        else:
            await send_gc_msg(f"PASSED - PMT Analog High Range Test")

        ################################################################################################################

        if not phd_test_error:
            return f"PHD test successful. Continue with next DUT."
        else:
            return f"PHD test failed. Check output and log files for details."

    finally:
        await fmb_endpoint.SetDigitalOutput(FMBDigitalOutput.PSUON, 1)   # Base Tester PSUON is active low


async def phd_high_voltage_test(hv_target, average=phd_voltage_average):

    hv_voltages = await phd_get_hv_voltages(hv_target, average)
    if GlobalVar.get_stop_gc():
        return f"phd_high_voltage_test stopped by user"

    hv_test_error = False

    PyLogger.logger.info(f"Measurment Operation Voltages (HV = {hv_target}):")
    PyLogger.logger.info(f"voltage    ; value     ; reference ; error")
    for voltage, value in hv_voltages['hv_gate_1'].items():
        reference = phd_get_hv_reference(voltage, hv_target, hv_gate=1)
        tolerance = phd_voltage_tolerance[voltage]
        error = abs((value - reference) / reference) if (reference != 0) else float('NaN')
        PyLogger.logger.info(f"{voltage:10} ; {value:9.2f} ; {reference:9.2f} ; {error:7.2%}")
        if (reference != 0 and error > tolerance) or (reference == 0 and abs(value) > phd_high_voltage_off_max_abs):
            hv_test_error = True

    PyLogger.logger.info(f"Blocking Operation Voltages (HV = {hv_target}):")
    PyLogger.logger.info(f"voltage    ; value     ; reference ; error")
    for voltage, value in hv_voltages['hv_gate_0'].items():
        reference = phd_get_hv_reference(voltage, hv_target, hv_gate=0)
        tolerance = phd_voltage_tolerance[voltage]
        error = abs((value - reference) / reference) if (reference != 0) else float('NaN')
        PyLogger.logger.info(f"{voltage:10} ; {value:9.2f} ; {reference:9.2f} ; {error:7.2%}")
        if (reference != 0 and error > tolerance) or (reference == 0 and abs(value) > phd_high_voltage_off_max_abs):
            hv_test_error = True
    
    if hv_test_error:
        return 'FAILED'
    else:
        return 'PASSED'


def phd_get_hv_reference(voltage, hv_target=0, hv_gate=0):
    if voltage in phd_voltage_reference:
        return phd_voltage_reference[voltage]
    if voltage == 'hv_monitor':
        return hv_target
    if (hv_gate == 0) and (voltage in phd_block_dynode_coefficient):
        coefficient = phd_block_dynode_coefficient[voltage]
    else:
        coefficient = phd_dynode_coefficient[voltage]
    if hv_target < -10:
        return ((hv_target + 10) * coefficient - 10)
    else:
        return hv_target


async def phd_get_voltage(name, average=1):
    name = name.lower()
    result = 0.0
    if name in ptb_control_sw_off:
        input = ptb_control_sw_off[name]
        await eef_endpoint.SetDigitalOutput(EEFDigitalOutput.PTBCONTROLSWITCH1, 0)
        await asyncio.sleep(0.1)
        for i in range(average):
            result += (await eef_endpoint.GetAnalogInput(input['channel']))[0] * input['scaling'] + input['offset']
        return result / average
    if name in ptb_control_sw_on:
        input = ptb_control_sw_on[name]
        await eef_endpoint.SetDigitalOutput(EEFDigitalOutput.PTBCONTROLSWITCH1, 1)
        await asyncio.sleep(0.1)
        for i in range(average):
            result += (await eef_endpoint.GetAnalogInput(input['channel']))[0] * input['scaling'] + input['offset']
        return result / average
    if name in ptb_cathode_sw_off:
        input = ptb_cathode_sw_off[name]
        await eef_endpoint.SetDigitalOutput(EEFDigitalOutput.PTBCATHODESWITCH1, 0)
        await asyncio.sleep(0.1)
        for i in range(average):
            result += (await eef_endpoint.GetAnalogInput(input['channel']))[0] * input['scaling'] + input['offset']
        return result / average
    if name in ptb_cathode_sw_on:
        input = ptb_cathode_sw_on[name]
        await eef_endpoint.SetDigitalOutput(EEFDigitalOutput.PTBCATHODESWITCH1, 1)
        await asyncio.sleep(0.1)
        for i in range(average):
            result += (await eef_endpoint.GetAnalogInput(input['channel']))[0] * input['scaling'] + input['offset']
        return result / average
    if name in ptb_anode_sw_on:
        input = ptb_anode_sw_on[name]
        await eef_endpoint.SetDigitalOutput(EEFDigitalOutput.PTBANODESWITCH1, 1)
        await asyncio.sleep(0.1)
        for i in range(average):
            result += (await eef_endpoint.GetAnalogInput(input['channel']))[0] * input['scaling'] + input['offset']
        return result / average
    if name == 'hv_monitor':
        input = phd_hv_monitor
        for i in range(average):
            result += (await meas_endpoint.GetParameter(MeasurementParameter.PMT1HighVoltageMonitor))[0] * input['scaling'] + input['offset']
        return result / average
    
    raise ValueError(f"Invalid name")


async def phd_get_control_voltages():
    results = {}

    for voltage in ptb_control_sw_off:
        results[voltage] = await phd_get_voltage(voltage)
        if GlobalVar.get_stop_gc():
            return f"phd_get_control_voltages stopped by user"

    for voltage in ptb_control_sw_on:
        results[voltage] = await phd_get_voltage(voltage)
        if GlobalVar.get_stop_gc():
            return f"phd_get_control_voltages stopped by user"

    return results


async def phd_get_hv_voltages(hv_target, average=1):
    hv_voltages = ['hv_monitor', 'cathode', 'dyn1', 'dyn2', 'dyn3', 'dyn4', 'dyn5', 'dyn6', 'dyn7', 'dyn8', 'dyn9', 'anode']
    results = {'hv_gate_0':{}, 'hv_gate_1':{}}
    try:
        hv_setting = hv_target * phd_hv_setting['scaling'] + phd_hv_setting['offset']
        await meas_endpoint.SetParameter(MeasurementParameter.PMT1HighVoltageSetting, hv_setting)
        await meas_endpoint.SetParameter(MeasurementParameter.PMT1HighVoltageEnable, 1)
        await asyncio.sleep(1.0)
        
        await phd_set_hv_gate(0)
        await asyncio.sleep(0.1)
        for voltage in hv_voltages:
            results['hv_gate_0'][voltage] = await phd_get_voltage(voltage, average)
            if GlobalVar.get_stop_gc():
                return f"phd_get_hv_voltages stopped by user"
        
        await phd_set_hv_gate(1)
        await asyncio.sleep(0.1)
        for voltage in hv_voltages:
            results['hv_gate_1'][voltage] = await phd_get_voltage(voltage, average)
            if GlobalVar.get_stop_gc():
                return f"phd_get_hv_voltages stopped by user"

    finally:
        await meas_endpoint.SetParameter(MeasurementParameter.PMT1HighVoltageEnable, 0)

    return results


async def phd_set_hv_gate(enable):
    op_id = 'phd_set_hv_gate'
    seq_gen = meas_seq_generator()
    if (enable == 1):
        seq_gen.SetSignals(OutputSignal.HVGatePMT1)
    else:
        seq_gen.ResetSignals(OutputSignal.HVGatePMT1)
    seq_gen.Stop(0)
    meas_unit.ClearOperations()
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await meas_unit.ExecuteMeasurement(op_id)


async def phd_counting_measurment_test():

    dl_start = 0.0
    dl_stop = 1.0
    dl_step = 0.01

    dl_frequency = 0.25                     # Measure signal frequency at this dl

    frequency_reference = 125e6             # Signal frequency in Hz
    frequency_tolerance = 0.02              # 2% tolerance for relative error

    amplitude_min = dl_frequency + 0.01     # amplitude must be greater then dl_frequency
    
    noise_cps = 50.0                        # Only CPS values above this value are counted as noise TODO
    noise_limit = 0.25                      # No noise for dl values > amplitude + noise_limit TODO
    
    window_ms = 100

    dl_range = np.arange(dl_start, (dl_stop + 1e-6), dl_step).round(6)
    
    cps = np.zeros_like(dl_range)   # Counts Per Second
    dt = np.zeros_like(dl_range)    # Dead Time

    await eef_endpoint.SetDigitalOutput(EEFDigitalOutput.PTBANODESWITCH1, 0)
    await asyncio.sleep(0.1)

    op_id = 'pmt1_counting_measurement'
    meas_unit.ClearOperations()
    await load_pmt1_counting_measurement(op_id, window_ms)

    PyLogger.logger.info(f"Counting Measurement Test:")
    PyLogger.logger.info(f"dl    ; cps       ; dt")

    for i, dl in enumerate(dl_range):
        if GlobalVar.get_stop_gc():
            return f"phd_counting_measurment_test stopped by user"
        
        await meas_endpoint.SetParameter(MeasurementParameter.PMT1DiscriminatorLevel, dl)
        await asyncio.sleep(0.1)
        await meas_unit.ExecuteMeasurement(op_id)
        results = await meas_unit.ReadMeasurementValues(op_id)

        cps[i] = (results[0]  + (results[1]  << 32)) / window_ms * 1000.0
        dt[i]  = (results[2]  + (results[3]  << 32)) / window_ms * 1000.0

        PyLogger.logger.info(f"{dl:.3f} ; {cps[i]:9.0f} ; {dt[i]:9.0f}")

    frequency = cps[dl_range == dl_frequency][0]
    frequency_error = abs((frequency - frequency_reference) / frequency_reference)

    PyLogger.logger.info(f"frequency: {frequency / 1e6:.2f} MHz, frequency_error: {frequency_error:.2%}")

    amplitude = dl_range[cps >= (frequency_reference * (1 - frequency_tolerance))]
    amplitude = np.max(amplitude) if len(amplitude) else 0.0

    PyLogger.logger.info(f"amplitude: {amplitude:.3f} FS")

    noise_max = dl_range[cps > noise_cps]
    noise_max = np.max(noise_max) if len(noise_max) else 1.0

    PyLogger.logger.info(f"noise_max: {noise_max} FS")

    if frequency_error > frequency_tolerance or amplitude < amplitude_min or noise_max > (amplitude + noise_limit):
        return 'FAILED'
    else:
        return 'PASSED'


async def load_pmt1_counting_measurement(op_id, window_ms):

    window_us = round(window_ms * 1000)
    window_us_coarse, window_us_fine = divmod(window_us, 65536)

    us_tick_delay = 100     # 1 us

    seq_gen = meas_seq_generator()

    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)  # pmt1_cnt_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)  # pmt1_cnt_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)  # pmt1_dt_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)  # pmt1_dt_msb

    seq_gen.TimerWaitAndRestart(us_tick_delay)
    seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    if window_us_coarse > 0:
        seq_gen.Loop(window_us_coarse)
        seq_gen.Loop(65536)
        seq_gen.TimerWaitAndRestart(us_tick_delay)
        seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.LoopEnd()
        seq_gen.LoopEnd()
    if window_us_fine > 0:
        seq_gen.Loop(window_us_fine)
        seq_gen.TimerWaitAndRestart(us_tick_delay)
        seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.LoopEnd()

    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT1, deadTime=False, relative=False, resetCounter=False, cumulative=True, dword=True, addrPos=0, resultPos=0)
    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT1, deadTime=True, relative=False, resetCounter=True, cumulative=True, dword=True, addrPos=0, resultPos=2)

    seq_gen.LoopEnd()
    seq_gen.Stop(0)

    meas_unit.resultAddresses[op_id] = range(0, 4)
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)


async def phd_analog_measurement_test(mode='auto'):

    sample_interval_us = {'auto':20, 'low':20, 'high':2000}
    sample_count = 50

    integrator_mode = {'auto':IntegratorMode.integrate_autorange, 'low':IntegratorMode.integrate_in_low_range, 'high':IntegratorMode.integrate_in_high_range}

    await eef_endpoint.SetDigitalOutput(EEFDigitalOutput.PTBANODESWITCH1, 1)
    await asyncio.sleep(0.1)

    op_id = 'pmt1_analog_measurement'
    meas_unit.ClearOperations()
    await load_pmt1_analog_scan(op_id, sample_interval_us[mode], sample_count, integrator_mode[mode])
    await meas_unit.ExecuteMeasurement(op_id)
    results = await meas_unit.ReadMeasurementValues(op_id)

    time_us = np.zeros(sample_count + 1)
    analog_low = np.zeros(sample_count + 1)
    analog_high = np.zeros(sample_count + 1)

    high_range = (mode == 'high')

    PyLogger.logger.info(f"Analog Measurement Test (integrator_mode={mode}):")
    PyLogger.logger.info(f"time_us ; analog_low  ; analog_high")
    PyLogger.logger.info(f"      0 ;           0 ;           0")

    for i in range(sample_count):
        time_us[i + 1] = np.round((i + 1) * sample_interval_us[mode])
        analog_low[i + 1] = results[i * 2 + 2] - results[0] if results[i * 2 + 2] > 0 else 0
        analog_high[i + 1] = results[i * 2 + 3] - results[1] if results[i * 2 + 3] > 0 else 0

        # Ignore first high range value to ensure the charge transfer is complete.
        if not high_range and results[i * 2 + 3] > 0:
            high_range = True
            analog_high[i + 1] = 0

        PyLogger.logger.info(f"{time_us[i + 1]:7.0f} ; {analog_low[i + 1]:11.0f} ; {analog_high[i + 1]:11.0f}")

    al_offset = results[0]
    al_max = np.max(analog_low)
    al_slope = 0.0
    al_intercept = 0.0
    al_r_squared = 0.0

    linear_range = np.logical_and((analog_low > 0), (analog_low <= (al_max * 0.9)))
    if mode in ['low', 'auto'] and np.any(linear_range):
        al_intercept, al_slope = polyfit(time_us[linear_range], analog_low[linear_range], deg=1)
        al_r_squared = np.corrcoef(time_us[linear_range], analog_low[linear_range])[0,1] ** 2
    
    ah_offset = results[1]
    ah_max = np.max(analog_high)
    ah_slope = 0.0
    ah_intercept = 0.0
    ah_r_squared = 0.0

    linear_range = np.logical_and((analog_high > 0), (analog_high <= (ah_max * 0.9)))
    if mode in ['high', 'auto'] and np.any(linear_range):
        ah_intercept, ah_slope = polyfit(time_us[linear_range], analog_high[linear_range], deg=1)
        ah_r_squared = np.corrcoef(time_us[linear_range], analog_high[linear_range])[0,1] ** 2

    analog_test_error = False

    # al_offset within 10% of 100mV offset voltage
    al_offset_error = abs((al_offset - 1310) / 1310)
    if al_offset_error > 0.1:
        analog_test_error = True

    PyLogger.logger.info(f"al_offset: {al_offset:.0f}, al_offset_error: {al_offset_error:.2%}")

    # ah_offset within 10% of 100mV offset voltage
    ah_offset_error = abs((ah_offset - 1310) / 1310)
    if ah_offset_error > 0.1:
        analog_test_error = True

    PyLogger.logger.info(f"ah_offset: {ah_offset:.0f}, ah_offset_error: {ah_offset_error:.2%}")

    if mode == 'auto':
        # al_max within 10% of FS/2 - 100mV offset voltage
        al_max_error = abs((al_max - 31458) / 31458)
        if al_max_error > 0.1:
            analog_test_error = True

        PyLogger.logger.info(f"al_max: {al_max:.0f}, al_max_error: {al_max_error:.2%} (mode = 'auto')")

        # ah_max within 10% of ah_slope reference * measurement time
        ah_max_error = abs((ah_max - 1190) / 1190)
        if ah_max_error > 0.1:
            analog_test_error = True

        PyLogger.logger.info(f"ah_max: {ah_max:.0f}, ah_max_error: {ah_max_error:.2%} (mode = 'auto')")

        # al_slope within 10% of empirical value
        al_slope_error = abs((al_slope - 120) / 120)
        if al_slope_error > 0.1:
            analog_test_error = True

        PyLogger.logger.info(f"al_slope: {al_slope:.1f}, al_slope_error: {al_slope_error:.2%} (mode = 'auto')")

        # ah_slope within 10% of al_slope reference / 101
        ah_slope_error = abs((ah_slope - 1.19) / 1.19)
        if ah_slope_error > 0.1:
            analog_test_error = True

        PyLogger.logger.info(f"ah_slope: {ah_slope:.3f}, ah_slope_error: {ah_slope_error:.2%} (mode = 'auto')")

        # al_intercept less or equal to 0.1% of FS
        if al_intercept > 66:
            analog_test_error = True

        PyLogger.logger.info(f"al_intercept: {al_intercept:.0f} (mode = 'auto')")

        # ah_intercept less or equal to 0.1% of FS
        if ah_intercept > 66:
            analog_test_error = True

        PyLogger.logger.info(f"ah_intercept: {ah_intercept:.0f} (mode = 'auto')")

        # al_r_squared better or equal linearity than r² = 0.99995
        if al_r_squared < 0.99995:
            analog_test_error = True

        PyLogger.logger.info(f"al_r_squared: {al_r_squared:.6f} (mode = 'auto')")

        # ah_r_squared better or equal linearity than r² = 0.99995
        if ah_r_squared < 0.99995:
            analog_test_error = True

        PyLogger.logger.info(f"ah_r_squared: {ah_r_squared:.6f} (mode = 'auto')")

    if mode == 'low':
        # al_max within 10% of FS - 100mV offset voltage
        al_max_error = abs((al_max - 64226) / 64226)
        if al_max_error > 0.1:
            analog_test_error = True

        PyLogger.logger.info(f"al_max: {al_max:.0f}, al_max_error: {al_max_error:.2%} (mode = 'low')")

        # ah_max must be 0
        if ah_max != 0:
            analog_test_error = True

        PyLogger.logger.info(f"ah_max: {ah_max:.0f} (mode = 'low')")

        # al_slope within 10% of empirical value
        al_slope_error = abs((al_slope - 120) / 120)
        if al_slope_error > 0.1:
            analog_test_error = True

        PyLogger.logger.info(f"al_slope: {al_slope:.1f}, al_slope_error: {al_slope_error:.2%} (mode = 'low')")

        # al_intercept less or equal to 0.1% of FS
        if al_intercept > 66:
            analog_test_error = True

        PyLogger.logger.info(f"al_intercept: {al_intercept:.0f} (mode = 'low')")

        # al_r_squared better or equal linearity than r² = 0.99999
        if al_r_squared < 0.99999:
            analog_test_error = True

        PyLogger.logger.info(f"al_r_squared: {al_r_squared:.6f} (mode = 'low')")

    if mode == 'high':
        # al_max must be 0
        if al_max != 0:
            analog_test_error = True

        PyLogger.logger.info(f"al_max: {al_max:.0f} (mode = 'high')")

        # ah_max within 10% of FS - 100mV offset voltage
        ah_max_error = abs((ah_max - 64226) / 64226)
        if ah_max_error > 0.1:
            analog_test_error = True

        PyLogger.logger.info(f"ah_max: {ah_max:.0f}, ah_max_error: {ah_max_error:.2%} (mode = 'high')")

        # ah_slope within 10% of al_slope reference / 101
        ah_slope_error = abs((ah_slope - 1.19) / 1.19)
        if ah_slope_error > 0.1:
            analog_test_error = True

        PyLogger.logger.info(f"ah_slope: {ah_slope:.3f}, ah_slope_error: {ah_slope_error:.2%} (mode = 'high')")

        # ah_intercept less or equal to 0.1% of FS
        if ah_intercept > 66:
            analog_test_error = True

        PyLogger.logger.info(f"ah_intercept: {ah_intercept:.0f} (mode = 'high')")

        # ah_r_squared better or equal linearity than r² = 0.99999
        if ah_r_squared < 0.99999:
            analog_test_error = True

        PyLogger.logger.info(f"ah_r_squared: {ah_r_squared:.6f} (mode = 'high')")

    if analog_test_error:
        return 'FAILED'
    else:
        return 'PASSED'


async def plot_analog_signal_scan(time_us, analog_low, analog_high, file_name='graph.png'):

    plt.clf()

    plt.title('PMT Analog Signal Scan')
    plt.xlabel('time [µs]')
    plt.ylabel('Analog Signal')
    plt.plot(time_us, analog_low, label='analog_low', color='b')
    plt.plot(time_us, analog_high, label='analog_high', color='r')
    plt.legend(loc='upper right')

    plt.savefig(os.path.join(report_path, file_name))
    await send_gc_event('RefreshGraph', file_name=file_name)


async def load_pmt1_analog_scan(op_id, sample_interval_us=0.0, sample_count=1, integrator_mode=IntegratorMode.integrate_autorange):
    
    conversion_delay_us = 12    #  12 us
    conversion_delay = round(conversion_delay_us * 100)

    full_reset_delay = 40000    # 400 us
    range_switch_delay = 25     # 250 ns
    reset_switch_delay = 2000   #  20 us
    us_tick_delay = 100         # 1 us

    sample_interval_us = round(sample_interval_us - conversion_delay_us)
    sample_interval_us_coarse, sample_interval_us_fine = divmod(sample_interval_us, 65536)

    seq_gen = meas_seq_generator()

    # results = [pmt1_alo, pmt1_aho, pmt1_al, pmt1_ah, pmt1_al, pmt1_ah, ...]
    seq_gen.SetAddrReg(relative=False, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=0)
    seq_gen.Loop(2 + sample_count * 2)
    seq_gen.ClearResultBuffer(relative=True, dword=False, addrReg=0, addr=0)
    seq_gen.SetAddrReg(relative=True, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=1)
    seq_gen.LoopEnd()

    seq_gen.SetAddrReg(relative=False, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=0)

    seq_gen.ResetSignals(OutputSignal.InputGatePMT1)
    seq_gen.SetSignals(OutputSignal.HVGatePMT1)
    
    seq_gen.SetAnalogControl(pmt1=AnalogControlMode.full_offset_reset)

    seq_gen.TimerWaitAndRestart(full_reset_delay)
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.full_reset)

    seq_gen.TimerWaitAndRestart(conversion_delay)
    seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1)

    seq_gen.TimerWaitAndRestart(range_switch_delay)
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.low_range_reset)
    seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=False, dword=False, addrPos=0, resultPos=1)

    seq_gen.TimerWaitAndRestart(reset_switch_delay)
    seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1)
    seq_gen.SetIntegratorMode(pmt1=integrator_mode)

    seq_gen.TimerWaitAndRestart(conversion_delay)
    seq_gen.SetSignals(OutputSignal.InputGatePMT1)

    if sample_interval_us_coarse > 0:
        seq_gen.Loop(sample_interval_us_coarse)
        seq_gen.Loop(65536)
        seq_gen.TimerWaitAndRestart(us_tick_delay)
        seq_gen.LoopEnd()
        seq_gen.LoopEnd()
    if sample_interval_us_fine > 0:
        seq_gen.Loop(sample_interval_us_fine)
        seq_gen.TimerWaitAndRestart(us_tick_delay)
        seq_gen.LoopEnd()

    seq_gen.TimerWaitAndRestart(conversion_delay)
    seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=0)
    seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1)

    seq_gen.SetAddrReg(relative=False, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=2)

    if sample_count > 1:
        seq_gen.Loop(sample_count - 1)

        if sample_interval_us_coarse > 0:
            seq_gen.Loop(sample_interval_us_coarse)
            seq_gen.Loop(65536)
            seq_gen.TimerWaitAndRestart(us_tick_delay)
            seq_gen.LoopEnd()
            seq_gen.LoopEnd()
        if sample_interval_us_fine > 0:
            seq_gen.Loop(sample_interval_us_fine)
            seq_gen.TimerWaitAndRestart(us_tick_delay)
            seq_gen.LoopEnd()

        seq_gen.TimerWaitAndRestart(conversion_delay)
        seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=True, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=0)
        seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=True, ignoreRange=False, isHiRange=True, addResult=False, dword=False, addrPos=0, resultPos=1)
        seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1)

        seq_gen.SetAddrReg(relative=True, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=2)

        seq_gen.LoopEnd()

    seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=True, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=True, ignoreRange=False, isHiRange=True, addResult=False, dword=False, addrPos=0, resultPos=1)

    seq_gen.ResetSignals(OutputSignal.InputGatePMT1)
    seq_gen.ResetSignals(OutputSignal.HVGatePMT1)
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.full_reset)
    seq_gen.Stop(0)

    meas_unit.resultAddresses[op_id] = range(0, (2 + sample_count * 2))
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

